/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	VillagerPlans
**
** DESCRIPTION:	Villager plans
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef VILLAGER_PLANS_H
#define VILLAGER_PLANS_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include "AnimManager.h"
#include "Behavior.h"
#include "SkillTypes.h"
#include "../Collectables/Collectable.h"
#include "../Effects/FloatingAnim.h"
#include "../Game/Isola.h"
#include "../Puzzles/Puzzle.h"
#include "../Sound/EnvironmentalSound.h"
#include "../Utils/theStringManager.h"
#include "../World/ContentMap.h"
#include "../World/Waypoint.h"

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/
const int VillagerPlans_c_iMaxActionPlans = 80;
const int VillagerPlans_c_iMaxBehaviorDataSize = 32;

/******************************************************************************\
** TYPE DEFINITIONS
\******************************************************************************/
enum EAction
{
	eAction_DoNothing = 0,

	eAction_Wander,
	eAction_Wait,
	eAction_Go,
	eAction_FollowPath,
	eAction_Swim,
	eAction_Dance,
	eAction_Learn,
	eAction_Work,
	eAction_Climb,
	eAction_Dive,
	eAction_Jump,
	eAction_Bend,
	eAction_Embrace,
	eAction_Hide,
	eAction_Carry,
	eAction_Drop,
	eAction_AdvanceState,
	eAction_SpawnAnim,
	eAction_SpawnAnimAtVillager,
	eAction_SpawnRare,
	eAction_ShakeHead,
	eAction_Heal,
	eAction_SoundOn,
	eAction_SoundOff,
	eAction_TakeCorpse,
	eAction_DealerSay,
	eAction_AdvancePuzzle,
	eAction_ResetPuzzle,
	eAction_Cheer,
	eAction_Twirl_CW,
	eAction_Twirl_CCW,
	eAction_Disco,
	eAction_DoTheMonkey,
	eAction_Revel,
	eAction_LieDown,
	eAction_Scrub,
	eAction_Exorcist,
	eAction_Joy_Twirl_CW,
	eAction_PlayAnim,
	eAction_Pushups,

	eAction_NewBehavior,

	eAction_NumberOf
};

enum EPriority
{
	ePriority_Normal = 0,
	ePriority_High,

	ePriority_NumberOf
};

enum ESpeed
{
	eSpeed_Warp = 0,
	eSpeed_Crawl = 40,
	eSpeed_Walk = 100,
	eSpeed_Run = 150,
	eSpeed_Sprint = 300
};

enum EBodyPosition
{
	eBodyPosition_Upright = 0,
	eBodyPosition_Bent,
	eBodyPosition_Sitting,
	eBodyPosition_Working,
	eBodyPosition_Hidden,
	eBodyPosition_Still,
	eBodyPosition_Climbing,
	eBodyPosition_Diving,
	eBodyPosition_Jumping,
	eBodyPosition_Lying,
	eBodyPosition_PointingNE,
	eBodyPosition_PointingSE,
	eBodyPosition_PointingSW,
	eBodyPosition_PointingNW,
	eBodyPosition_ReachingS,
	eBodyPosition_ReachingN,
	eBodyPosition_Watching,

	eBodyPosition_NumberOf
};

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CVillagerPlans
{
public:
	
	//class structures
	struct SAdvancePuzzleData
	{
		EPuzzle				m_ePuzzle;
		int					m_iRef;
	};
	
	struct SAdvanceStateData
	{
		EWorldState			m_eState;
	};
	
	struct SFloatingAnimData
	{
		EFloatingAnim		m_eAnim;
		int					m_iXPos;
		int					m_iYPos;
	};

	struct SCarryData
	{
		ECarrying			m_eCarrying;
	};

	struct SClimbData
	{
		int					m_iHeight;
	};
	
	struct SDiveData
	{
		EAnim				m_eAnim;
		int					m_iHeight;
		EDirection			m_eDirection;
		int					m_iLandingOffset;
		float				m_fLaunchVelocity;
	};

	struct SFaceDirectionData
	{
		EDirection			m_eDirection;
		EHeadDirection		m_eHeadDirection;
	};
	
	struct SGoingData
	{
		ESpeed				m_eSpeed;
		int					m_iTargetX;
		int					m_iTargetY;
		int					m_iCounter;
	};

	struct SJumpData
	{
		int					m_iHeight;
		float				m_fLaunchVelocity;
	};
	
	struct SLearnData
	{
		ESkill				m_eSkill;
	};
	
	struct SNewBehaviorData
	{
		EBehavior			m_eBehavior;
		unsigned char		m_uiData[VillagerPlans_c_iMaxBehaviorDataSize];
	};

	struct SPlayAnimData
	{
		EAnim				m_eAnim;
		bool				m_bIsReversed;
		float				m_fSpeed;
	};

	struct SSayData
	{
		StringId			m_eString;
	};
	
	struct SSoundOnData
	{
		ESound				m_eSound;
		float				m_fVolume;
		ESoundType			m_eSoundType;
	};
	
	struct SSwimData
	{
		bool				m_bOnBack;
	};

	struct STakeCorpseData
	{
		int					m_iCorpse;
	};
	
	struct SWaitData
	{
		EBodyPosition		m_eBodyPosition;
		EDirection			m_eDirection;
		EHeadDirection		m_eHeadDirection;
	};
	
	struct SWanderingData
	{
		ESpeed				m_eSpeed;
		int					m_iTargetX;
		int					m_iTargetY;
	};
	
	union UData
	{
		SAdvancePuzzleData	m_sAdvancePuzzleData;
		SAdvanceStateData	m_sAdvanceStateData;
		SFloatingAnimData	m_sFloatingAnimData;
		SCarryData			m_sCarryData;
		SClimbData			m_sClimbData;
		SDiveData			m_sDiveData;
		SFaceDirectionData	m_sFaceDirectionData;
		SGoingData			m_sGoingData;
		SJumpData			m_sJumpData;
		SLearnData			m_sLearnData;
		SNewBehaviorData	m_sNewBehaviorData;
		SPlayAnimData		m_sPlayAnimData;
		SSayData			m_sSayData;
		SSoundOnData		m_sSoundOnData;
		SSwimData			m_sSwimData;
		STakeCorpseData		m_sTakeCorpseData;
		SWaitData			m_sWaitData;
		SWanderingData		m_sWanderingData;
	};
	
	struct SActionPlan
	{
		//structure data
		EAction				m_eAction;
		UData				m_uData;
		unsigned int		m_uiTimer;

		//structure methods
		//setup
		public:		SActionPlan() 
					{ 
						m_eAction = eAction_DoNothing; 
						m_uiTimer = 0; 
					}
	};
	
	//class data
	SActionPlan				m_sActionPlan[VillagerPlans_c_iMaxActionPlans];

	//class methods
	//process
	private:	void AddPlan(const SActionPlan sPlan, const EPriority ePriority);
	public:		void AdvanceCurrentPlan(CVillager &Villager);
	public:		void NextPlan(CVillager &Villager, bool startNextPlan = true);
	public:		void ForgetPlans(CVillager &Villager);
	public:		void StartNewBehavior(CVillager &Villager);
	public:		void ProcessCurrentPlan(CVillager &Villager);

	public:		void PlanToWander(const int iTime, const ESpeed eSpeed);
	public:		void PlanToWait(const int iTime, const EBodyPosition eBodyPosition, const EHeadDirection eHeadDirection);
	public:		void PlanToWait(const int iTime, const EBodyPosition eBodyPosition, const EDirection eDirection, const EHeadDirection eHeadDirection);	
	public:		void PlanToWait(const int iTime, const EBodyPosition eBodyPosition = eBodyPosition_Upright);

	public:		void PlanToGo(const CContentMap::EObject eObject, const ESpeed eSpeed = eSpeed_Walk, const EPriority ePriority = ePriority_Normal);
	public:		void PlanToGo(const EWaypoint eWaypoint, const ESpeed eSpeed = eSpeed_Walk, const EPriority ePriority = ePriority_Normal) ;
	public:		void PlanToGo(const EWaypoint eWaypoint, const ldwPoint sOffset, const ESpeed eSpeed = eSpeed_Walk, const EPriority ePriority = ePriority_Normal);
	public:		void PlanToGo(const ldwPoint sPos, const ESpeed eSpeed = eSpeed_Walk, const EPriority ePriority = ePriority_Normal);
	public:		void PlanToGo(const int iTargetX, const int iTargetY, const ESpeed eSpeed = eSpeed_Walk, const EPriority ePriority = ePriority_Normal);
	public:		void PlanToSwim(const int iTime, const bool bOnBack);
	public:		void PlanToDance(const int iTime);
	public:		void PlanToLearn(const ESkill eSkill);
	public:		void PlanToWork(const int iTime);
	public:		void PlanToClimb(const int iHeight);
	public:		void PlanToDive(const EAnim eAnim, const int iHeight, const EDirection eDirection, const int iLandingOffset);
	public:		void PlanToJump(const int iHeight = -1);
	public:		void PlanToBend(const int iTime);
	public:		void PlanToCheer(const int iTime);
	public:		void PlanToTwirlCW(const int iTime); 
	public:		void PlanToTwirlCCW(const int iTime);
	public:		void PlanToExorcist(const int iTime);
	public:		void PlanToJoyTwirlCW(const int iTime);
	public:		void PlanToDisco(const int iTime);
	public:		void PlanToDoTheMonkey(const int iTime);
	public:		void PlanToRevel(const int iTime);
	public:		void PlanToLieDown(const int iTime);
	public:     void PlanToScrub(const int iTime);
	public:		void PlanToEmbrace();
	public:		void PlanToHide(const int iTime);
	public:		void PlanToCarry(const ECarrying eCarrying);
	public:		void PlanToDrop();
	public:		void PlanToAdvanceState(const EWorldState eWorldState);
	public:		void PlanToAdvancePuzzle(const EPuzzle ePuzzle, const int iRef = -1);
	public:		void PlanToResetPuzzle(const EPuzzle ePuzzle); 

	public:		void PlanToShakeHead(const int iTime, const EBodyPosition eBodyPosition);
	public:		void PlanToHeal(const EPriority ePriority = ePriority_Normal);
	public:		void PlanToPlaySound(const ESound eSound, const float fVolume = 1.0f, const ESoundType eSoundType = eSoundType_OneShot);
	public:		void PlanToStopSound();
	public:		void PlanToTakeCorpse(const int iCorpse);
	public:		void PlanToSay(const StringId eStringId);
	public:		void PlanToFollowPath(const int iTargetX, const int iTargetY, const ESpeed eSpeed);
	public:		void PlanToPlayAnim(const int iTime, const EAnim eAnim, const bool bIsReversed = false, const float fSpeed = 0.0f);	
	public:		void PlanNewBehavior(const EBehavior eBehavior, const SBehaviorData &sData = SBehaviorData());
	public:		void PlanToDoPushups(const int iTime);
	public: 	void PlanToSpawnAnim(const EFloatingAnim eAnim, int xPos, int yPos);
	public: 	void PlanToSpawnAnimAtVillager(const EFloatingAnim eAnim, int xPos, int yPos);
	public: 	void PlanToSpawnRareCollectible();

	//access
	public:		const bool IsCarrying() const { return m_sActionPlan[0].m_eAction == eAction_Carry; }
	public:		const bool IsGoing() const { return m_sActionPlan[0].m_eAction == eAction_Go; }
	public:		const bool IsIdling() const { return m_sActionPlan[0].m_eAction == eAction_DoNothing; }
	public:		const bool IsWaiting() const { return m_sActionPlan[0].m_eAction == eAction_Wait; }
	public:		const bool HasCurrentPlanExpired() const;
	public:		const EAction CurrentAction() const  { return m_sActionPlan[0].m_eAction; }
	public:		const EAction NextAction() const  { return m_sActionPlan[1].m_eAction; }
};

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/